<?php

namespace App\Controllers\Clients;

use App\Controllers\BaseController;
use App\Models\ClientModel;
use App\Models\ActivityModel;
use App\Models\PotentialModel;
use App\Models\StatusModel;
use App\Models\ProductModel;

class Clients extends BaseController
{
    protected $clientModel;
    protected $activityModel;
    protected $potentialModel;
    protected $statusModel;
    protected $productModel;

    public function __construct()
    {
        $this->clientModel    = new ClientModel();
        $this->activityModel  = new ActivityModel();
        $this->potentialModel = new PotentialModel();
        $this->statusModel    = new StatusModel();
        $this->productModel   = new ProductModel();
    }

    // 🧾 Списък с клиенти
    public function index()
    {
        $clients = $this->clientModel->orderBy('id', 'DESC')->findAll();
        return view('Clients/list', [
            'title'   => 'Всички клиенти',
            'clients' => $clients,
        ]);
    }

    // ➕ Създаване
    public function create()
    {
        return view('Clients/form', [
            'title'      => 'Добавяне на нов клиент',
            'activities' => $this->activityModel->orderBy('name')->findAll(),
            'potentials' => $this->potentialModel->orderBy('name')->findAll(),
            'statuses'   => $this->statusModel->orderBy('name')->findAll(),
            'products'   => $this->productModel->orderBy('name')->findAll(),
            'agents'     => $this->getAgents(),
        ]);
    }

    // 💾 Запис
    public function save()
    {
        $data = [
            'firm_name'   => $this->request->getPost('firm_name'),
            'potential'   => $this->request->getPost('potential'),
            'activity'    => $this->request->getPost('activity'),
            'status'      => $this->request->getPost('status'),
            'products'    => $this->request->getPost('products'),
            'contact'     => $this->request->getPost('contact'),
            'client_type' => $this->request->getPost('client_type'),
            'salesperson' => $this->request->getPost('salesperson'),
            'notes'       => $this->request->getPost('notes'),
        ];

        $id = $this->clientModel->insert($data);
        return redirect()->to("/clients/preview/$id");
    }

    // 👁️ Преглед
    public function preview($id)
    {
        $client = $this->clientModel->find($id);
        if (!$client) return redirect()->to('/clients')->with('error', 'Клиентът не е намерен.');

        return view('Clients/preview', [
            'title' => 'Преглед на клиент',
            'data'  => $client,
        ]);
    }

    // ✏️ Редакция
    public function edit($id)
    {
        $client = $this->clientModel->find($id);
        if (!$client) return redirect()->to('/clients')->with('error', 'Клиентът не е намерен.');

        return view('Clients/form', [
            'title'      => 'Редакция на клиент',
            'client'     => $client,
            'activities' => $this->activityModel->orderBy('name')->findAll(),
            'potentials' => $this->potentialModel->orderBy('name')->findAll(),
            'statuses'   => $this->statusModel->orderBy('name')->findAll(),
            'products'   => $this->productModel->orderBy('name')->findAll(),
            'agents'     => $this->getAgents(),
        ]);
    }

    // 💾 Обновяване
    public function update($id)
    {
        $data = $this->request->getPost();
        $this->clientModel->update($id, $data);
        return redirect()->to("/clients/preview/$id")->with('success', 'Клиентът е обновен успешно!');
    }

    // ❌ Изтриване
    public function delete($id)
    {
        $this->clientModel->delete($id);
        return redirect()->to('/clients')->with('success', 'Клиентът е изтрит.');
    }

    // 🧩 Вземане на агенти (fallback ако няма offers)
    private function getAgents()
    {
        try {
            $db = \Config\Database::connect();
            $query = $db->table('offers')->select('agent')->distinct()->orderBy('agent')->get()->getResultArray();
            return $query ?: [];
        } catch (\Exception $e) {
            return [
                ['agent' => 'Илия Стилев'],
                ['agent' => 'Радослав Радев'],
                ['agent' => 'Младен Мавродиев'],
                ['agent' => 'Станимира Богданова']
            ];
        }
    }

    public function products()
    {
        $method = $this->request->getMethod(true); // 'GET' | 'POST' | 'PUT' | 'DELETE'
    
        switch ($method) {
            case 'GET': {
                $rows = $this->productModel->select('id, name')->orderBy('name')->findAll();
                return $this->response->setJSON($rows);
            }
    
            case 'POST': {
                $name = trim((string) $this->request->getPost('name'));
                if ($name === '') {
                    return $this->response->setJSON(['success' => false, 'error' => 'Празно име.']);
                }
    
                // анти-дубликат, ако има UNIQUE(name)
                $exists = $this->productModel->where('name', $name)->first();
                if ($exists) {
                    return $this->response->setJSON(['success' => true, 'id' => (int)$exists['id'], 'name' => $exists['name']]);
                }
    
                $id = $this->productModel->insert(['name' => $name], true);
                return $this->response->setJSON(['success' => true, 'id' => (int)$id, 'name' => $name]);
            }
    
            case 'PUT': {
                // при PUT ползвай raw input
                $data = $this->request->getRawInput();
                $id   = (int)($data['id'] ?? 0);
                $name = trim((string)($data['name'] ?? ''));
    
                if ($id <= 0 || $name === '') {
                    return $this->response->setJSON(['success' => false, 'error' => 'Липсва ID или име.']);
                }
    
                // не позволявай да се преименува на съществуващо име
                $dup = $this->productModel->where('name', $name)->where('id !=', $id)->first();
                if ($dup) {
                    return $this->response->setJSON(['success' => false, 'error' => 'Вече има продукт с това име.']);
                }
    
                $this->productModel->update($id, ['name' => $name]);
                return $this->response->setJSON(['success' => true]);
            }
    
            case 'DELETE': {
                // вземи id от body или query (?id=)
                $raw = $this->request->getRawInput();
                $id  = (int)($raw['id'] ?? $this->request->getGet('id') ?? 0);
    
                if ($id <= 0) {
                    return $this->response->setJSON(['success' => false, 'error' => 'Липсва валидно ID.']);
                }
    
                // винаги изтривай с WHERE, за да няма: "Deletes are not allowed..."
                $this->productModel->where('id', $id)->delete();
                return $this->response->setJSON(['success' => true]);
            }
        }
    
        return $this->response->setStatusCode(405)->setJSON(['success' => false, 'error' => 'Методът не е позволен.']);
    }
    
    public function searchFirms()
    {
        $term = trim($this->request->getGet('q') ?? '');
        $db = \Config\Database::connect();
    
        $builder = $db->table('_klient')
            ->select('klient_id as id, klient_name as text')
            ->where('klient_name !=', '');
    
        if ($term !== '') {
            $builder->like('klient_name', $term);
        }
    
        $results = $builder->orderBy('klient_name', 'ASC')
                           ->limit(50)
                           ->get()
                           ->getResultArray();
    
        return $this->response->setJSON($results);
    }
    
    public function addOption()
{
    $type = $this->request->getPost('type');
    $name = trim($this->request->getPost('name'));

    if (!$type || $name === '') {
        return $this->response->setJSON(['success' => false, 'error' => 'Липсват данни.']);
    }

    // модела според типа
    $modelMap = [
        'activity'  => $this->activityModel,
        'potential' => $this->potentialModel,
        'status'    => $this->statusModel,
    ];

    if (!isset($modelMap[$type])) {
        return $this->response->setJSON(['success' => false, 'error' => 'Невалиден тип.']);
    }

    $model = $modelMap[$type];

    // Проверка за дублиране
    $exists = $model->where('name', $name)->first();
    if ($exists) {
        return $this->response->setJSON([
            'success' => true,
            'name'    => $exists['name'],
        ]);
    }

    // Запис
    $model->insert(['name' => $name]);

    return $this->response->setJSON([
        'success' => true,
        'name'    => $name,
    ]);
}


}
