<?php

namespace App\Controllers\Dashboard;

use App\Controllers\BaseController;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class Debtors extends BaseController
{
    // ===============================
    // PAGE VIEW
    // ===============================
    public function index()
    {
        return view('Dashboard/debtors', [
            'title' => 'Длъжници ВАЛПЕРС'
        ]);
    }

    // ===============================
    // AJAX DATA LOADER (JSON)
    // ===============================
    public function load()
    {
        include APPPATH . 'Controllers/Dashboard/DebtorsData.php';
    
        $mode = $this->request->getGet('mode') ?? 'all';
        $data = $debtorsData;
    
        if (!$data) {
            return $this->response->setJSON(['items' => []]);
        }
    
        $today = new \DateTime();
    
        // АЛГОРИТЪМ ЗА АВТОМАТИЧНО ПОПЪЛВАНЕ НА ТИП ДОКУМЕНТ И ПРОТОКОЛ
        foreach ($data as &$row) {

            if (
                stripos($row['Тип'] ?? '', 'кредитно') !== false ||
                stripos($row['Документ'] ?? '', 'К') === 0
            ) {
                $row['Тип документ'] = 'Кредитно известие';
            }
            // 2) Фактура — има фактура
            elseif (!empty($row['Фак.No:'])) {
                $row['Тип документ'] = 'Фактура';
            }
            // 3) Протокол — НЯМА фактура
            else {
                $row['Тип документ'] = 'Протокол';
            }

    
            // Автоматично генериране на номер на протокол ако липсва
            if (empty($row['Протокол']) && $row['Тип документ'] === 'Протокол') {
                // Генериране на номер на протокол във формат: П-YYYY-MM-DD-XXXX
                $dateObj = \DateTime::createFromFormat('m/d/Y', $row['От дата']);
                if (!$dateObj) {
                    $dateObj = new \DateTime($row['От дата']);
                }
                
                $randomSuffix = str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
                $row['Протокол'] = 'П-' . $dateObj->format('Y-m-d') . '-' . $randomSuffix;
            }
    
            // Ако типът е фактура, но няма номер на документ - генерираме такъв
            if (empty($row['Документ']) && $row['Тип документ'] === 'Фактура') {
                $dateObj = \DateTime::createFromFormat('m/d/Y', $row['От дата']);
                if (!$dateObj) {
                    $dateObj = new \DateTime($row['От дата']);
                }
                
                $randomSuffix = str_pad(mt_rand(1000, 9999), 4, '0', STR_PAD_LEFT);
                $row['Документ'] = 'Ф-' . $dateObj->format('Ym') . '-' . $randomSuffix;
            }
    
            // Останалата логика за падежи...
            $row['dateObj'] = \DateTime::createFromFormat('m/d/Y', $row['От дата']);
            if (!$row['dateObj']) {
                $row['dateObj'] = new \DateTime($row['От дата']);
            }
    
            $row['year']       = (int)$row['dateObj']->format('Y');
            $row['month']      = $this->bgMonth($row['dateObj']->format('n'));
            $row['date_short'] = $row['dateObj']->format('d-m');
    
            // ПАДЕЖ ЛОГИКА
            $diffDays = (int)$today->diff($row['dateObj'])->days;
    
            $limit = 14; // default
            if (stripos($row['Плащане'], 'КАСА') !== false) {
                $limit = 7;
            }
    
            $dueDate = (clone $row['dateObj'])->modify("+{$limit} days");
            $statusText = 'В ПАДЕЖ';
            $statusCode = 'good';
    
            if ($today > $dueDate) {
                $overdueDays = $today->diff($dueDate)->days;
                $statusText = "Просрочено ({$overdueDays} дни)";
                $statusCode = 'bad';
            }
    
            $row['ПАДЕЖ']        = $statusText;
            $row['limit_days']   = $limit;
            $row['due_date']     = $dueDate->format('d.m.Y');
            $row['due_date_iso'] = $dueDate->format('Y-m-d');
            $row['days_diff']    = $diffDays;
            $row['overdue_days'] = $today > $dueDate ? $today->diff($dueDate)->days : 0;
            $row['status_code']  = $statusCode;
        }
        unset($row);
    
        // FILTERS by MODE (cash/bank/all) + GROUPING
        switch ($mode) {
            case 'cash':
                $filtered = array_filter($data, fn($r) =>
                    mb_stripos($r['Плащане'], 'КАСА') !== false
                );
                $result = $this->groupByFirm($filtered);
                break;
    
            case 'bank':
                $filtered = array_filter($data, fn($r) =>
                    mb_stripos($r['Плащане'], 'БАНКА') !== false
                );
                $result = $this->groupByFirm($filtered);
                break;
    
            default: // all
                $result = $this->groupByAgent($data);
                break;
        }
    
        return $this->response->setJSON(['items' => $result]);
    }

    public function export()
    {
        include APPPATH . 'Controllers/Dashboard/DebtorsData.php';
    
        $mode = $this->request->getGet('mode') ?? 'all';
        $firm = $this->request->getGet('firm') ?? '';
        $agent = $this->request->getGet('agent') ?? '';
        $email = $this->request->getGet('email') ?? '';
        $docType = $this->request->getGet('docType') ?? '';
        $status = $this->request->getGet('status') ?? '';
        $from = $this->request->getGet('from') ?? '';
        $to = $this->request->getGet('to') ?? '';
    
        $data = $debtorsData;
        if (!$data) die("Няма данни за експорт.");
    
        $today = new \DateTime();
    
        // =============================
        // 1) ПЪЛНА ОБРАБОТКА НА РЕДОВЕ
        // =============================
        foreach ($data as &$row) {
    
            // --- ТИП ДОКУМЕНТ
            if (
                stripos($row['Тип'] ?? '', 'кредитно') !== false ||
                stripos($row['Документ'] ?? '', 'К') === 0
            ) {
                $row['Тип документ'] = 'Кредитно известие';
            }
            elseif (!empty($row['Фак.No:']) || !empty($row['Документ'])) {
                $row['Тип документ'] = 'Фактура';
            }
            else {
                $row['Тип документ'] = 'Протокол';
            }
    
            // Дата
            $row['dateObj'] = \DateTime::createFromFormat('m/d/Y', $row['От дата'])
                               ?: new \DateTime($row['От дата']);
    
            // Падеж
            $limit = stripos($row['Плащане'], 'КАСА') !== false ? 7 : 14;
    
            $due = (clone $row['dateObj'])->modify("+{$limit} days");
    
            if ($today > $due) {
                $days = $today->diff($due)->days;
                $row['ПАДЕЖ'] = "Просрочено ({$days} дни)";
            } else {
                $row['ПАДЕЖ'] = "В ПАДЕЖ";
            }
    
            $row['due_date'] = $due->format('d.m.Y');
        }
        unset($row);
    
        // =============================
        // 2) ФИЛТРИРАНЕ
        // =============================
        $filtered = array_filter($data, function($r) use ($firm, $agent, $email, $docType, $status, $from, $to, $mode) {
    
            if ($firm && $r['Фирма'] !== $firm) return false;
            if ($agent && $r['Агент'] !== $agent) return false;
            if ($email && $r['e-mail'] !== $email) return false;
            if ($docType && $r['Тип документ'] !== $docType) return false;
    
            if ($status === 'good' && strpos($r['ПАДЕЖ'], 'Просрочено') !== false) return false;
            if ($status === 'bad' && strpos($r['ПАДЕЖ'], 'Просрочено') === false) return false;
    
            if ($from && $r['dateObj'] < new \DateTime($from)) return false;
            if ($to && $r['dateObj'] > new \DateTime($to)) return false;
    
            if ($mode === 'cash' && mb_stripos($r['Плащане'], 'КАСА') === false) return false;
            if ($mode === 'bank' && mb_stripos($r['Плащане'], 'БАНКА') === false) return false;
    
            return true;
        });
    
        if (empty($filtered)) die("Няма данни за експорт.");
    
        // =============================
        // 3) EXCEL – СТРУКТУРА КАТО ТВОЯТА
        // =============================
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle("Длъжници");
    
        // === Заглавие
        $title = "Длъжници " . ($firm ?: "ВАЛПЕРС");
    
        if ($mode === 'bank') $title .= " ПО БАНКИ";
        if ($mode === 'cash') $title .= " В БРОЙ";
    
        $sheet->mergeCells("A1:I1");
        $sheet->setCellValue("A1", $title);
        $sheet->getStyle("A1")->getFont()->setBold(true)->setSize(15)->getColor()->setARGB("FFFFFFFF");
        $sheet->getStyle("A1")->getFill()->setFillType("solid")->getStartColor()->setARGB("FF1D4ED8");
        $sheet->getStyle("A1")->getAlignment()->setHorizontal("center");
    
        // === HEADER КАТО ПРИМЕРА
        $headers = [
            "A2" => "От дата",
            "B2" => "Тип Документ",
            "C2" => "Документ",
            "D2" => "Фирма",
            "E2" => "Сума",
            "F2" => "Плащане",
            "G2" => "Агент",
            "H2" => "Email",
            "I2" => "Падеж",
        ];
    
        foreach ($headers as $cell => $text) {
            $sheet->setCellValue($cell, $text);
            $sheet->getStyle($cell)->getFont()->setBold(true);
            $sheet->getStyle($cell)->getAlignment()->setHorizontal("center");
            $sheet->getStyle($cell)->getFill()->setFillType("solid")
                   ->getStartColor()->setARGB("FFE0F2FE");
        }
    
        // === ROWS
        $row = 3;
        $total = 0;
    
        foreach ($filtered as $r) {
    
            $sheet->setCellValue("A{$row}", $r["От дата"]);
            $sheet->setCellValue("B{$row}", $r["Тип документ"]);
            $sheet->setCellValue("C{$row}", $r["Документ"] ?? $r["Протокол"]);
            $sheet->setCellValue("D{$row}", $r["Фирма"]);
            $sheet->setCellValue("E{$row}", $r["за сума"]);
            $sheet->setCellValue("F{$row}", $r["Плащане"]);
            $sheet->setCellValue("G{$row}", $r["Агент"]);
            $sheet->setCellValue("H{$row}", $r["e-mail"]);
            $sheet->setCellValue("I{$row}", $r["ПАДЕЖ"]);
    
            // цвят за просрочено
            if (strpos($r["ПАДЕЖ"], "Просрочено") !== false) {
                $sheet->getStyle("I{$row}")->getFill()->setFillType("solid")
                      ->getStartColor()->setARGB("FFFFC4C4");
            }
    
            $total += $this->convertAmount($r["за сума"]);
            $row++;
        }
    
        // === ОБЩ БАЛАНС
        $sheet->mergeCells("A{$row}:D{$row}");
        $sheet->setCellValue("A{$row}", "ОБЩ БАЛАНС");
        $sheet->getStyle("A{$row}")->getFont()->setBold(true)->getColor()->setARGB("FF000000");
        $sheet->getStyle("A{$row}")->getFill()->setFillType("solid")
              ->getStartColor()->setARGB("FFFFFF00");
    
        $sheet->setCellValue("E{$row}", number_format($total, 2, ',', ' ') . " лв.");
        $sheet->getStyle("E{$row}")->getFont()->setBold(true);
        $sheet->getStyle("E{$row}")->getFill()->setFillType("solid")
              ->getStartColor()->setARGB("FFFFFF00");
    
        // Auto-size
        foreach (range('A','I') as $col)
            $sheet->getColumnDimension($col)->setAutoSize(true);
    
        // Output
        header("Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        header("Content-Disposition: attachment; filename=\"debtors_export.xlsx\"");
        header("Cache-Control: max-age=0");
    
        $writer = new Xlsx($spreadsheet);
        $writer->save("php://output");
        exit;
    }


    // GROUPING FOR ALL (agent groups)
    private function groupByAgent($data)
    {
        $out = [];

        foreach ($data as $row) {
            $agent = $row['Агент'];

            if (!isset($out[$agent])) {
                $out[$agent] = [
                    'group_name' => $agent,
                    'rows'  => [],
                    'total' => 0
                ];
            }

            $out[$agent]['rows'][] = $row;
            $out[$agent]['total'] += $this->convertAmount($row['за сума']);
        }

        return array_values($out);
    }

    // GROUPING FOR CASH / BANK (firm groups)
    private function groupByFirm($data)
    {
        $out = [];

        foreach ($data as $row) {
            $firm = $row['Фирма'];

            if (!isset($out[$firm])) {
                $out[$firm] = [
                    'group_name' => $firm,
                    'rows'  => [],
                    'total' => 0
                ];
            }

            $out[$firm]['rows'][] = $row;
            $out[$firm]['total'] += $this->convertAmount($row['за сума']);
        }

        return array_values($out);
    }

    // Convert "1 234,56 лв." -> float
    private function convertAmount($str)
    {
        $str = str_replace([' ', 'лв.', ','], ['', '', '.'], $str);
        return floatval($str);
    }

    private function bgMonth($m)
    {
        $names = [
            1=>'яну',2=>'фев',3=>'мар',4=>'апр',5=>'май',6=>'юни',
            7=>'юли',8=>'авг',9=>'сеп',10=>'окт',11=>'ное',12=>'дек'
        ];
        return $names[(int)$m];
    }
}
