<?php

namespace App\Libraries\Gensoft;

/*
 * ------------------------------------------------------------
 * GENSOFT FIREBIRD – SCHEMA & TRIGGERS CHECK
 * ------------------------------------------------------------

  Този скрипт:
  - проверява достъпност до Gensoft Firebird сървъра
  - създава Firebird връзка
  - проверява и създава липсващи колони / индекси / генератор
  - открива дали STOKI.ITEM_ID е липсвала (критично състояние)
  - създава нужните тригери
  ------------------------------------------------------------ */

//
class LIB__FirebirdServices {

    protected $conn;

    public function __construct() {
        helper('firebird');
        $this -> nL = (php_sapi_name() === 'cli' ? PHP_EOL : '<br>');
    }

    /**  Проверка дали тригерите съществуват
      ------------------------------------------- */
    public function hasTriggers($conn): bool {
        echo $this -> nL;
        echo "▶ Проверка на тригери ...\n" . $this -> nL;

        if (h_trigger_exists($conn, 'STOKI_ITEM_ID_BI')) {
            echo "ℹ️️ Тригерът STOKI_ITEM_ID_BI вече съществува.\n" . $this -> nL;

            return true;
        }

        return false;
    }

    /** SCHEMA / DDL
      ------------------------------------------------------ */
    public function columns_and_Generator($conn): bool {
        echo $this -> nL;
        echo "▶ Проверка и създаване на колони, индекси и генератор...\n" . $this -> nL;

        if (h_column_exists($conn, 'STOKI', 'ITEM_ID')) {
            echo "ℹ️ Колоната ITEM_ID вече съществува в STOKI.\n" . $this -> nL;
            return true;
        }

        $schema = [
            [
                'table'  => 'STOKI',
                'column' => 'ITEM_ID',
                'index'  => 'STOKI_IDX_ITEM_ID',
            ],
            [
                'table'  => 'SER_NO',
                'column' => 'ITEM_ID',
                'index'  => 'SERNO_IDX_ITEM_ID',
            ],
        ];

        $tr = ibase_trans(IBASE_WRITE | IBASE_COMMITTED | IBASE_REC_VERSION | IBASE_WAIT, $conn);

        try {
            foreach ($schema as $def) {
                $col   = $def['column'];
                $table = $def['table'];
                $index = $def['index'];

                if (!h_column_exists($conn, $table, $col)) {
                    ibase_query($tr, "ALTER TABLE $table ADD $col INTEGER");
                    echo "✔ Добавена колона $table . $col\n" . $this -> nL;
                }

                if (!empty($index) && !h_index_exists($conn, $index)) {
                    ibase_query($tr, "CREATE INDEX $index ON $table ($col)");
                    echo "✔ Създаден индекс $index\n" . $this -> nL;
                }
            }

            if (!h_generator_exists($conn, 'GEN_UNIVERSAL_ID')) {
                ibase_query($tr, "CREATE GENERATOR GEN_UNIVERSAL_ID");
                echo "✔ Създаден генератор GEN_UNIVERSAL_ID\n" . $this -> nL;
            }

            ibase_commit($tr);

            echo "✅ Schema колони, индекси и генератор приключи успешно\n\n" . $this -> nL . $this -> nL;
            return true;
        } catch (\Throwable $e) {
            ibase_rollback($tr);
            return false;
        }
    }

    /** TRIGGERS
      ------------------------------------------- */
    public function triggers($conn): bool {
        echo $this -> nL;
        echo "▶ Проверка и създаване на тригери ...\n" . $this -> nL;

        if (h_trigger_exists($conn, 'STOKI_ITEM_ID_BI')) {
            echo "ℹ Тригерът STOKI_ITEM_ID_BI вече съществува.\n" . $this -> nL;
            return true;
        }

        echo "❌ Тригерът STOKI_ITEM_ID_BI не съществува. Създаваме всички нужни тригери...\n" . $this -> nL;

        $tr = ibase_trans(IBASE_WRITE | IBASE_COMMITTED | IBASE_REC_VERSION | IBASE_WAIT, $conn);

        try {
            $sql = "EXECUTE BLOCK AS
                        BEGIN
                            EXECUTE STATEMENT '
                                CREATE TRIGGER STOKI_DEF_BI0 BEFORE INSERT ON STOKI_DEF
                                AS
                                BEGIN
                                    if (new.item_id is null) then
                                         new.item_id = (select max(STOKI_DEF.ITEM_ID) + 1 from STOKI_DEF);
                                END';

                            EXECUTE STATEMENT '
                                CREATE TRIGGER STOKI_ITEM_ID_BI BEFORE INSERT ON STOKI
                                AS
                                BEGIN
                                   IF (NEW.ITEM_ID IS NULL) THEN 
                                        NEW.ITEM_ID = (SELECT sd.ITEM_ID FROM STOKI_DEF sd WHERE sd.STOKA = NEW.STOKA AND sd.GRUPA = NEW.GRUPA);
                                END';

                            EXECUTE STATEMENT '
                                CREATE TRIGGER SERNO_ITEM_ID_BI BEFORE INSERT ON SER_NO
                                AS
                                BEGIN
                                   IF (NEW.ITEM_ID IS NULL) THEN 
                                        NEW.ITEM_ID = (SELECT sd.ITEM_ID  FROM STOKI_DEF sd WHERE sd.STOKA = NEW.STOKA AND sd.GRUPA = NEW.GRUPA);
                                END';
                        END;";

            ibase_query($tr, $sql);
            ibase_commit($tr);

            echo "ℹ️ Тригерите STOKI_DEF_BI0, STOKI_ITEM_ID_BI, SERNO_ITEM_ID_BI са създадени.\n" . $this -> nL;
            return true;
        } catch (\Throwable $e) {
            ibase_rollback($tr);
            return false;
        }
    }

    /** GENERATOR SYNC
      ---------------------------------------------------- */
    public function syncGeneratorWithTable($trans, string $table, string $column, string $generator = 'GEN_UNIVERSAL_ID'): void {

        $sql = "
            EXECUTE BLOCK AS
              DECLARE max_id INTEGER;
            BEGIN
              SELECT COALESCE(MAX($column), 0)
                FROM $table
                INTO :max_id;

              IF (GEN_ID($generator, 0) < max_id) THEN
                EXECUTE STATEMENT
                  'ALTER SEQUENCE $generator RESTART WITH ' || (max_id + 1);
            END
        ";

        $res = ibase_query($trans, $sql);

        if ($res === false) {
            throw new \RuntimeException(ibase_errmsg());
        }
    }

    /** Проверка дали ITEM_ID е новосъздадена и попълване на колоните
      --------------------------------------------------------------- */
    public function populate_itemId_firmId($conn): bool {
        echo $this -> nL;
        echo "▶ Попълване на новите колони ITEM_ID...\n" . $this -> nL;

        if (h_column_exists($conn, 'STOKI', 'ITEM_ID')) {
            echo "ℹ️ ️Попълването е прекратено защото вече са пълни.\n" . $this -> nL;
            return true;
        }

        $trans = ibase_trans(IBASE_WRITE | IBASE_COMMITTED | IBASE_REC_VERSION | IBASE_WAIT, $conn);

        // синхронизираме генератора
        $this -> syncGeneratorWithTable($trans, 'STOKI_DEF', 'ITEM_ID');

        /* UPDATE FIRMI */

        ibase_query($trans, " UPDATE STOKI_DEF SET ITEM_ID = GEN_ID(GEN_UNIVERSAL_ID, 1) WHERE ITEM_ID IS NULL; ");

        $sql = <<<SQL
                    EXECUTE BLOCK
                    AS
                    BEGIN
                        UPDATE STOKI s
                           SET ITEM_ID = (
                               SELECT sd.ITEM_ID
                               FROM STOKI_DEF sd
                               WHERE sd.STOKA = s.STOKA
                                 AND sd.GRUPA = s.GRUPA
                                 AND sd.ITEM_ID IS NOT NULL
                               ROWS 1
                           )
                         WHERE s.ITEM_ID IS NULL;

                        UPDATE SER_NO s
                           SET ITEM_ID = (
                               SELECT sd.ITEM_ID
                               FROM STOKI_DEF sd
                               WHERE sd.STOKA = s.STOKA
                                 AND sd.GRUPA = s.GRUPA
                                 AND sd.ITEM_ID IS NOT NULL
                               ROWS 1
                           )
                         WHERE s.ITEM_ID IS NULL;
                    END
               SQL;

        $result = ibase_query($trans, $sql);

        if ($result === false) {
            ibase_rollback($trans);
            die('Firebird error: ' . ibase_errmsg());
        }

        ibase_commit($trans);

        echo '✅ OK – обновяването на колоните ITEM_ID приключи успешно' . $this -> nL;
        return true;
    }

}
