<?php

namespace App\Modules\zenova\Models;

use \App\Models\BaseModel as BaseModel;

class MODEL__zenova_oferta extends BaseModel {

    private $currentZenovaId = 0;

    /** Set current Zenova list ID (used as implicit context)
      @param int $id
      @return void
      --------------------------------------------------- */
    public function setCurrentZenovaId($id) {
        $this -> currentZenovaId = (int) $id;
    }

    /** Get currently active Zenova list ID
      @return int
      --------------------------------------------------- */
    private function getCurrentZenovaId() {
        return $this -> currentZenovaId;
    }

    /** Count products in Zenova list (used for pagination)
      @param array    $productIds
      @param int|null $zenovaListaId
      @return int
      ---------------------------------------------------------------------- */
    function count__products_fromZenova($productIds = [], $zenovaListaId = null) {
        // Guard: empty product set
        if (empty($productIds)) {
            return 0;
        }

        // Resolve Zenova list ID if not explicitly provided
        if ($zenovaListaId === null) {
            $zenovaListaId = $this -> getCurrentZenovaId();
        }

        // Guard: no Zenova context
        if (empty($zenovaListaId)) {
            return 0;
        }

        return $this -> db -> table(self::TBL_PRODUCT . ' p')
                        -> join(self::TBL_PRODUCT_IN_ZENOVALISTA . ' pz',
                                'p.product_id = pz.product_id', 'inner'
                        )
                        -> whereIn('p.product_id', $productIds)
                        -> where('pz.zenovaLista_id', $zenovaListaId)
                        -> countAllResults();
    }

    /////////////////////////////////////////////////////////////////
    // ИЗВЛИЧАНЕ НА ЦЕНОВА ЛИСТА ПО ID
    public function get_zenova_by_id($id) {

        $r = $this -> db -> table(self::TBL_ZENOVA_LISTA)
                        -> where('id', $id)
                        -> get() -> getRow();
//dd(json_decode($r -> colorRows, true) );

        return $r;
    }

    // ИЗВЛИЧАНЕ НА ВСИЧКИ ПРОДУКТИ ОТ ЦЕНОВА ЛИСТА ПО $data['productIds']
    public function get__products_fromZenova($data = [], $limit = 20, $offset = 0) {
        if (empty($data['productIds']) || empty($data['zenovaLista_id'])) {
            return [];
        }

        $productIds = array_map('intval', $data['productIds']);
        $search     = trim($data['search'] ?? '');

        $builder = $this -> db -> table(self::TBL_PRODUCT . ' p')
                -> join(self::TBL_PRODUCT_IN_ZENOVALISTA . ' pz',
                        'pz.product_id = p.product_id', 'inner')
                -> join(self::TBL_PRODUCT_PRICE_LEVEL . ' sp',
                        'sp.product_id = p.product_id', 'left')
                -> join(self::TBL_BRAND . ' b',
                        'b.brand_id = p.brand_id', 'left')
                -> join(self::TBL_MODEL . ' pm',
                        'pm.model_id = p.model_id', 'left')
                -> join(self::TBL_PRODUCT_SITES . ' ps',
                        'ps.product_id = p.product_id', 'left')
                -> whereIn('p.product_id', $productIds);

        if ($search !== '') {
            $builder -> groupStart()
                    -> like('p.product_name', $search)
                    -> orLike('p.kod', $search)
                    -> orLike('p.oem', $search)
                    -> orLike('p.ean', $search)
                    -> groupEnd();
        }

        $builder -> groupBy('p.product_id')
                -> orderBy('FIELD(p.product_id, ' . implode(',', $productIds) . ')')
                -> limit($limit, $offset);

        $this -> selectCols($builder, [
            'debug' => false,
            'only'  => [
                'pm' => ['model', 'model_id']
            ],
                //'skip'  => [],'extra' => []
        ]);

        return $builder -> get() -> getResultArray();
    }

    // всички продукти към сайтоиве маркирани
    public function get__prodAttr($productId = null) {
        return $this -> db -> table(self::TBL_PRODUCT_CHARACTERISTIC . ' p')
                        -> select('ca.value, pa.product_characteristic_text')
                        -> join(self::TBL_CATEGORY_ATTRIBUTE . ' ca', 'category_characteristic_id', 'left')
                        -> join(self::TBL_PRODUCT_CHAR_VALUE . ' pa', 'product_characteristic_value_id', 'left')
                        -> where('p.product_id', $productId)
                        -> groupBy('ca.value')
                        -> get() -> getResultArray();
    }

    /**
     * Get product category
     */
    public function get__productCategory($productId = null) {
        return $this -> db
                        -> table(self::TBL_PRODUCT . ' p')
                        -> select('c.category_name, c.parent_id')
                        -> join(self::TBL_CATEGORY . ' c', 'c.category_id = p.category_id', 'left')
                        -> where('p.product_id', $productId)
                        -> get()
                        -> getRowArray();
    }

    /**
     * Get product category with proper fallback:
     * - If subcategory exists → show it
     * - If not → show root category
     */
    public function get__productCategoryWithParent($productId = null) {
        $row = $this -> db
                -> table(self::TBL_PRODUCT . ' p')
                -> select('
                p.categoryRoot_id,
                p.category_id,
                c.category_name      AS sub_name,
                c.parent_id,
                cr.category_name     AS root_name
            ')
                -> join(self::TBL_CATEGORY . ' c', 'c.category_id = p.category_id', 'left')
                -> join(self::TBL_CATEGORY . ' cr', 'cr.category_id = p.categoryRoot_id', 'left')
                -> where('p.product_id', $productId)
                -> get()
                -> getRowArray();

        if (!$row) {
            return null;
        }

        // CASE 1: Has subcategory
        if (!empty($row['sub_name'])) {
            return [
                'category' => $row['sub_name'],
                'parent'   => $row['root_name'] ?: null,
            ];
        }

        // CASE 2: Only root category
        if (!empty($row['root_name'])) {
            return [
                'category' => $row['root_name'],
                'parent'   => null,
            ];
        }

        return null;
    }

    // продукт към всички сайтове ($isBulk == true всички продукти)
    public function product_toSites($productId = null, $is_toSite = false, $isBulk = false) {
        $newData       = [];
        $errorMessages = [];

        $productIds = is_array($productId) ? array_map('intval', $productId) : array_map('intval', explode(',', $productId));

        if (empty($productIds)) {
            return ['err' => 'Грешка: масивът с продукти е празен.'];
        }

        $sites = $this -> db -> table(self::TBL_SP_SITE)
                        -> select('sp_site_id')
                        -> get() -> getResultArray();

        if (empty($sites)) {
            return ['err' => 'Не са открити сайтове. От менюто [Настройки - общи настройки → Сайтове] се дефинират сайтовете, в които ще участват продуктите.'];
        }

        // ако са отметнати
        if ($is_toSite) {
            foreach ($productIds as $p) {
                foreach ($sites as $s) {
                    $newData[] = [
                        'product_id'      => $p,
                        'sp_site_id'      => (int) $s['sp_site_id'],
                        'visibility_type' => 0,
                    ];
                }
            }
        }

        $this -> db -> transBegin();

        if ($is_toSite) {
            if (!empty($newData)) {
                $this -> db -> query('SET FOREIGN_KEY_CHECKS=0');
                $upsertResult = $this -> db -> table(self::TBL_PRODUCT_SITES) -> upsertBatch($newData);
                $this -> db -> query('SET FOREIGN_KEY_CHECKS=1');

                if ($upsertResult === -1 || !$upsertResult) {
                    $errorMessages[] = 'Грешка: Неуспешно добавяне на продукти към сайтове. Направете запис на ценовата листа и опитайте отново.';
                }
            }
        } else {
            // Премахване от product_sites
            $deleteResult = $this -> db -> table(self::TBL_PRODUCT_SITES)
                    -> whereIn('product_id', $productIds)
                    -> delete();

            if ($deleteResult === -1 || !$deleteResult) {
                $errorMessages[] = 'Грешка: Неуспешно премахване на продукти от сайтове.';
            }

            // премахване на продуктите от незапазените поръчки
            $jsonRemoveSql = "JSON_REMOVE(product_json, " . implode(', ', array_map(fn($p) => "'$.$p'", $productIds)) . ")";

            $whereSql = implode(' OR ', array_map(fn($p) => "JSON_CONTAINS_PATH(product_json, 'one', '$.$p')", $productIds));

            $this -> db -> table(self::TBL_ORDERCART)
                    -> set('product_json', $jsonRemoveSql, false)
                    -> where($whereSql, null, false)
                    -> update();
        }

        if (!$this -> db -> transStatus()) {
            $this -> db -> transRollback();
            $msg = !empty($errorMessages) ? implode(';  ', $errorMessages) : '';
            return $this -> dbError($msg);
        }

        $this -> db -> transComplete();

        $action = $isBulk ? 'Избраните продукти са ' : 'Избраният продукт е ';
        $result = $action . ($is_toSite ? 'добавен' : 'премахнат') . ' към всички сайтове.';

        return ['success' => $result];
    }

    public function delete_products_from_zenova($productIds = [], $zenovaId = '') {
        $productIds = is_array($productIds) ? $productIds : [$productIds];

        $this -> db -> table(self::TBL_PRODUCT_IN_ZENOVALISTA)
                -> whereIn('product_id', $productIds)
                -> delete();

        $idsRegex = implode('|', array_map('intval', $productIds));
        $sql      = "TRIM(BOTH ',' FROM REGEXP_REPLACE(
                        REGEXP_REPLACE(productsID, 
                        '(?<=^|,)($idsRegex)(?=,|$)', ''),',+', ','))";

        $this -> db -> table(self::TBL_ZENOVA_LISTA)
                -> set('productsID', $sql, false)
                -> where('id', (int) $zenovaId)
                -> update();

        return;
    }

    public function save_oferta($id = null, $data = [], $deletedProductArrId = []) {
        $settings = service('settings');
        $general  = $settings -> get('App.general') ?? [];

        try {
            $this -> db -> transStart();

            if ($deletedProductArrId) {
                $this -> db -> table(self::TBL_PRODUCT_IN_ZENOVALISTA)
                        -> whereIn('product_id', $deletedProductArrId)
                        -> delete();
            }

            $builder = $this -> db -> table(self::TBL_ZENOVA_LISTA);

            // обновяване на продукти които ще са в ценовата листа
            if (is_numeric($id)) {
                $zenovaId = $id;

                $builder -> where('id', $id) -> update($data['zenova']);
            } else {
                if (!$builder -> insert($data['zenova'])) {
                    return $this -> dbError('Грешка при запис на нова ценова листа!');
                }

                $zenovaId = $this -> db -> insertID();
            }

            foreach ($data['zProduct'] as &$p) {
                $p['zenovaLista_id']   = $zenovaId;
                $p['zenovaLista_name'] = $data['zenova']['offersName'];
            }

            // ако са нови продукти които не пресъстват в никоя ценова листа
            $this -> db -> table(self::TBL_PRODUCT_IN_ZENOVALISTA)
                    -> upsertBatch($data['zProduct']);

            // DISCOUNT ЛОГИКА запис на ценообразуването в Талицата product_priceLevel
            $priceType = $general['zenoobr']['price_type'] ?? 'discount';
            if (in_array($priceType, ['discount', 'gensoft'])) {
                $productIds = array_column($data['zProduct'], 'product_id');

                $discounts = [
                    'A'    => (float) ($general['zenoobr']['A_discount'] ?? 0),
                    'B'    => (float) ($general['zenoobr']['B_discount'] ?? 0),
                    'Spec' => (float) ($general['zenoobr']['Spec_discount'] ?? 0),
                ];

                $this -> upsertDiscountPrices($productIds, $zenovaId, $discounts, $priceType);
            }

            if ($this -> db -> transStatus() === false) {
                return ['err' => 'Грешка при запис на ценовата листа!'];
            }

            $this -> db -> transComplete();

            return $zenovaId; // връща номера на ценовата листа
        } catch (\Exception $e) {
            $this -> db -> transRollback();

            // An exception occurred, handle the error
            return ['err' => 'Грешка: ' . $e -> getMessage()];
        }
    }

    public function delete_oferta($id) {
        $this -> db -> transBegin();

        try {
            $this -> db -> table(self::TBL_ZENOVA_LISTA)
                    -> where('id', $id)
                    -> delete();

            $this -> db -> transCommit();
        } catch (\Exception $e) {
            $this -> db -> transRollback();
            log_message('error', 'Transaction failed: ' . $e -> getMessage());
            return $e -> getMessage();
        }
    }

    /** Запис на ценообразуването в Талицата product_priceLevel
      ------------------------------------------------------------- */
    public function upsertDiscountPrices(array $productIds, int $zenovaId, array $discounts, string $priceType = 'discount') {
        if (empty($productIds)) {
            return;
        }

        $isGensoft   = ($priceType === 'gensoft');
        $now         = date('Y-m-d H:i:s');
        $productBase = [];
        $byProduct   = [];
        $batch       = [];

        // Вземаме съществуващите priceLevel записи  
        $rows = $this -> db -> table(self::TBL_PRODUCT_PRICE_LEVEL)
                        -> whereIn('product_id', $productIds)
                        -> get() -> getResultArray();

        foreach ($rows as $r) {
            $byProduct[$r['product_id']][$r['ofer_zenovaLista_id']] = $r;
        }

        // Вземаме базови цени + gensoft
        $products = $this -> db -> table(self::TBL_PRODUCT)
                        -> select('product_id, price_dostavna, gensoft_json')
                        -> whereIn('product_id', $productIds)
                        -> get() -> getResultArray();

        foreach ($products as $p) {

            if ($p['gensoft_json']) {
                $gensoftJson = json_decode($p['gensoft_json'], true);

                if (!empty($gensoftJson['KKC'])) {
                    $pid      = $p['product_id'];
                    $kkc      = (float) $gensoftJson['KKC'];
                    $promo    = (float) ($gensoftJson['promoPercent'] ?? 0);
                    $newPrice = $kkc * (1 - $promo / 100);

                    /* Ако $promo == 0 (без), $promo > 0(намаление), $promo < 0 (увеличение)

                      Случай 1 — Намаление (kkc = 5, promo = 20, $newPrice = 4)
                      KKC = 4,Zavishena = 5

                      Случай 1 — Увеличение (kkc = 5, promo = -20, $newPrice = 6)
                      KKC = 5, Zavishena = 6
                     */
                    $gensoftKKC[$pid]       = $promo > 0 ? $newPrice : $kkc;
                    $gensoftZavishena[$pid] = $promo !== 0 ? ($promo > 0 ? $kkc : $newPrice) : null;
                }
            }

            $productBase[$p['product_id']] = (float) str_replace(',', '.', $p['price_dostavna']);
        }

        foreach ($productIds as $productId) {
            $current = $byProduct[$productId][$zenovaId] ?? null;
            $base    = $byProduct[$productId][0] ?? null;

            $kkc = $current['cenaKKC'] ?? $base['cenaKKC'] ?? $productBase[$productId] ?? 0;

            if ($isGensoft) {
                $kkc              = $gensoftKKC[$productId] ?? null;
                $zavishenaKl_zena = $gensoftZavishena[$productId] ?? null;

                $discA    = $discB    = $discSpec = $kL       = null;
            } else {
                // предварително смятаме множители (по-бързо)
                $discA    = round($kkc * (1 - ($discounts['A'] ?? 0) / 100), 2);
                $discB    = round($kkc * (1 - ($discounts['B'] ?? 0) / 100), 2);
                $discSpec = round($kkc * (1 - ($discounts['Spec'] ?? 0) / 100), 2);

                $kL               = $kkc;
                $zavishenaKl_zena = null;
            }

            // ['A'=>10,'B'=>10,'Spec'=>5]
            $batch[] = [
                'product_id'          => $productId,
                'ofer_zenovaLista_id' => $zenovaId,
                'cenaA'               => $discA,
                'cenaB'               => $discB,
                'cenaSpec'            => $discSpec,
                'cenaKl'              => $kL,
                'cenaKKC'             => $kkc,
                'zavishenaKl_zena'    => $zavishenaKl_zena,
                'date_changed'        => $now
            ];
        }

        // UPSERT batch
        if ($batch) {
            $this -> db -> table(self::TBL_PRODUCT_PRICE_LEVEL) -> upsertBatch($batch);
        }
    }

}
