<?php

namespace App\Setups\Controllers;

use App\Controllers\BaseController;
use \App\Setups\Models\MODEL__preval;

class Preval extends BaseController {

    protected MODEL__preval $MODEL__preval;
    private $settings;

    public function __construct() {
        parent::__construct(); // 🔥 ТОВА Е КЛЮЧОВО
        $this -> settings = service('settings');

        $this -> MODEL__preval = new MODEL__preval();
    }

    public function save() {
        $general = service('settings') -> get("App.general") ?? [];
        $prev    = $general['preval'] ?? [];  // Старите preval настройки
        $new     = $this -> request -> getPost('settings')['preval'] ?? [];

        // Запази старата базова валута
        $new['old_base_currency'] = $prev['base_currency'] ?? 'BGN';
        $new['revalued']          = false;

        // Презапиши само ключа 'preval' без да сливаш с предишните
        $general['preval'] = $new;

        $this -> settings -> set('App.general', $general);

        return $this -> response -> setJSON([
                    "status" => "ok",
                    "msg"    => "Settings saved"
        ]);
    }

    public function checkAPI() {
        $url      = "https://www.ecb.europa.eu/stats/eurofxref/eurofxref-daily.xml";
        $response = @file_get_contents($url);

        return $this -> response -> setJSON([
                    "status" => $response ? "ok" : "error"
        ]);
    }

    /**
     * Load ECB rates for UI – JSON formatted
     */
    public function getRates() {
        $rates = $this -> getEcbRates();

        if (!$rates || !isset($rates['USD'])) {
            return $this -> response -> setJSON([
                        'status' => 'error',
                        'rates'  => null
            ]);
        }

        $EUR_BGN = 1.95583;
        $EUR_USD = $rates['USD'];
        $USD_BGN = $EUR_BGN / $EUR_USD;

        return $this -> response -> setJSON([
                    'status' => 'ok',
                    'rates'  => [
                        'EUR_BGN' => round($EUR_BGN, 4),
                        'USD_BGN' => round($USD_BGN, 4),
                        'EUR_USD' => round($EUR_USD, 4),
                    ]
        ]);
    }

    /**
     * Load ECB rates (raw) for internal DB conversion
     */
    private function getRatesRaw() {
        $rates = $this -> getEcbRates();

        if (!$rates || !isset($rates['USD'])) {
            return null;
        }

        $EUR_BGN = 1.95583;
        $EUR_USD = $rates['USD'];
        $USD_BGN = $EUR_BGN / $EUR_USD;

        return [
            'EUR_BGN' => $EUR_BGN,
            'USD_BGN' => $USD_BGN,
            'EUR_USD' => $EUR_USD,
        ];
    }

    public function getSettings() {
        $data = service('settings') -> get("App.general")['preval'] ?? [];

        return $this -> response -> setJSON([
                    'status' => 'ok',
                    'preval' => $data
        ]);
    }

    /**
     * Load raw CSV XML from ECB server
     */
    private function getEcbRates() {
        $url = "https://www.ecb.europa.eu/stats/eurofxref/eurofxref-daily.xml";

        $xmlString = @file_get_contents($url);
        if (!$xmlString)
            return null;

        $xml = simplexml_load_string($xmlString);
        if (!$xml)
            return null;

        $rates = [];

        foreach ($xml -> Cube -> Cube -> Cube as $c) {
            $rates[(string) $c['currency']] = (float) $c['rate'];
        }

        return $rates;
    }

    /**
     * Updates product base prices and price levels
     */
    public function updateDatabase() {
        // Allow only AJAX requests
        if (!$this -> request -> isAJAX()) {
            return $this -> response -> setJSON([
                        "error" => "not ajax"
            ]);
        }

        // Load general application settings
        $general = service("settings") -> get("App.general") ?? [];
        $preval  = $general["preval"] ?? [];

        if (!empty($preval['revalued'])) {
            return $this -> response -> setJSON([
                        "status" => "blocked",
                        "msg"    => "Превалутирането вече е изпълнено. Запазете настройките отново, за да го отключите."
            ]);
        }

        // Detect old and new base currencies
        $oldCurrency = $preval["old_base_currency"] ?? "BGN";
        $newCurrency = $preval["base_currency"] ?? "EUR";

        // Check for external system block (GENSOFT)
        if (!empty($preval["gensoft_block"])) {
            return $this -> response -> setJSON([
                        "status" => "blocked",
                        "msg"    => "GENSOFT BLOCK",
            ]);
        }

        // Load exchange rates (ECB or external source)
        $auto = $this -> getRatesRaw();
        if (!$auto) {
            return $this -> response -> setJSON([
                        "status" => "error",
                        "msg"    => "ECB NOT LOADED",
            ]);
        }

        // Normalize rates as floats
        $rates = [
            "EUR_BGN" => (float) $auto["EUR_BGN"],
            "EUR_USD" => (float) $auto["EUR_USD"],
        ];

        // -------------------------------------------------
        // MULTIPLIER CALCULATION
        // -------------------------------------------------
        // Determines how prices should be converted
        if ($oldCurrency === $newCurrency) {
            $mult = 1;
        } elseif ($oldCurrency === "BGN" && $newCurrency === "EUR") {
            $mult = 1 / $rates["EUR_BGN"]; // divide
        } elseif ($oldCurrency === "EUR" && $newCurrency === "BGN") {
            $mult = $rates["EUR_BGN"]; // multiply
        } elseif ($oldCurrency === "USD" && $newCurrency === "EUR") {
            $mult = 1 / $rates["EUR_USD"];
        } elseif ($oldCurrency === "EUR" && $newCurrency === "USD") {
            $mult = $rates["EUR_USD"];
        } else {
            return $this -> response -> setJSON([
                        "status" => "error",
                        "msg"    => "Unsupported conversion {$oldCurrency} → {$newCurrency}",
            ]);
        }

        // Sanity check for multiplier
        if ($mult <= 0) {
            return $this -> response -> setJSON([
                        "status" => "error",
                        "msg"    => "Invalid multiplier",
            ]);
        }


        // Обработка на базата
        try {
            $debug = $this -> MODEL__preval -> update_multi_dB($mult, $general);

            if (($debug['status'] ?? '') === 'ok') {
                $general['preval']['revalued'] = true;
                $this -> settings -> set('App.general', $general);
            }

            // Debug / verification response
            return $this -> response -> setJSON([
                        'status'       => $debug['status'],
                        "old_currency" => $oldCurrency,
                        "new_currency" => $newCurrency,
                        "rates"        => $rates,
                        "multiplier"   => $mult,
                        'db_debug'     => $debug
            ]);
        } catch (\Throwable $e) {
            return $this -> response -> setJSON([
                        "status" => "error",
                        "msg"    => $e -> getMessage(),
            ]);
        }
    }

    /**
     * Toggle Gensoft block for revaluation
     */
    public function gensoftToggle() {
        // Allow only AJAX requests
        if (!$this -> request -> isAJAX()) {
            return $this -> response -> setStatusCode(400) -> setJSON([
                        'status' => 'error',
                        'msg'    => 'Invalid request'
            ]);
        }

        // Load current settings
        $general = service('settings') -> get('App.general') ?? [];
        $preval  = $general['preval'] ?? [];

        // Toggle value
        $current = !empty($preval['gensoft_block']);
        $new     = !$current;

        // Save new value
        $preval['gensoft_block'] = $new;
        $general['preval']       = $preval;

        service('settings') -> set('App.general', $general);

        return $this -> response -> setJSON([
                    'status'        => 'ok',
                    'gensoft_block' => $new
        ]);
    }

}
