<?php

namespace App\Setups\Controllers;

use App\Controllers\BaseController;

class VariationRules extends BaseController {

    protected $db;

    public function __construct() {
        parent::__construct();
        $this -> db = \Config\Database::connect();
    }

    public function generate() {
        if (!$this -> request -> isAJAX()) {
            return $this -> response -> setJSON(["ok" => false]);
        }

        $productId = (int) $this -> request -> getPost("product_id");

        // CONFIG FROM UI
        $cfg = [
            "base_field" => $this -> request -> getPost("var_base_field"),
            "split_type" => $this -> request -> getPost("var_split_type"),
            "delimiter"  => $this -> request -> getPost("var_delimiter"),
            "substr_len" => (int) $this -> request -> getPost("var_substr"),
            "limit"      => (int) $this -> request -> getPost("var_limit"),
            "same_category"    => (int) $this -> request -> getPost(
                    "var_same_category"
            ),
            "same_subcategory" => (int) $this -> request -> getPost(
                    "var_same_subcategory"
            ),
            "same_brand"       => (int) $this -> request -> getPost("var_same_brand"),
            "same_model"       => (int) $this -> request -> getPost("var_same_model"),
            "only_active"      => (int) $this -> request -> getPost("var_only_active"),
            "min_timer"        => (int) $this -> request -> getPost("var_min_timer") * 60,
            "crawl_time"       =>
            (int) $this -> request -> getPost("var_crawl_time") * 60,
        ];

        // SAVE SETTINGS
        $settings                       = service("settings");
        $general                        = $settings -> get("App.general") ?? [];
        $general["variation_generator"] = $cfg;
        $settings -> set("App.general", $general);

        if (!$productId) {
            return $this -> response -> setJSON([
                        "ok"  => true,
                        "msg" => "Settings saved (no generation)",
            ]);
        }

        return $this -> response -> setJSON($this -> generateForProduct($productId));
    }

    private function generateForProduct(int $productId) {
        $cfg = service("settings") -> get("App.general")["variation_generator"] ??
                [];

        $base = $this -> db
                -> table("_product")
                -> where("product_id", $productId)
                -> get()
                -> getRowArray();

        if (!$base) {
            return ["ok" => false, "msg" => "Base product not found"];
        }

        $field = $cfg["base_field"] ?? "product_name";
        $value = trim($base[$field] ?? "");

        if ($value === "") {
            return ["ok" => false, "msg" => "Base field empty"];
        }

        if (($cfg["split_type"] ?? "") === "delimiter") {
            $delimiter = $cfg["delimiter"] ?? "-";
            $needle    = trim(explode($delimiter, $value)[0]);
        } else {
            $tokens = preg_split("/\s+/u", $value, -1, PREG_SPLIT_NO_EMPTY);
            $needle = "";

            for ($i = count($tokens); $i >= 2; $i--) {
                $try = implode(" ", array_slice($tokens, 0, $i));

                $cnt = $this -> db
                        -> table("_product")
                        -> where("product_id !=", $productId)
                        -> like($field, $try, "after")
                        -> countAllResults();

                if ($cnt > 0) {
                    $needle = $try;
                    break;
                }
            }

            if ($needle === "") {
                $needle = implode(" ", array_slice($tokens, 0, 2));
            }
        }

        if ($needle === "") {
            return ["ok" => false, "msg" => "Needle empty"];
        }

        $qb = $this -> db -> table("_product");
        $qb -> select("product_id, product_name");
        $qb -> like($field, $needle, "after");
        $qb -> where("product_id !=", $productId);

        if (!empty($cfg["only_active"])) {
            $qb -> groupStart()
                    -> where("Active", 1)
                    -> orWhere("Active IS NULL", null, false)
                    -> groupEnd();
        }

        if (!empty($cfg["same_category"]) && !empty($base["category_id"])) {
            $qb -> where("category_id", $base["category_id"]);
        }

        if (!empty($cfg["same_subcategory"]) && !empty($base["subCat_id"])) {
            $qb -> where("subCat_id", $base["subCat_id"]);
        }

        if (!empty($cfg["same_brand"]) && !empty($base["brand_id"])) {
            $qb -> where("brand_id", $base["brand_id"]);
        }

        if (!empty($cfg["same_model"]) && !empty($base["model_id"])) {
            $qb -> where("model_id", $base["model_id"]);
        }

        $qb -> limit((int) ($cfg["limit"] ?? 20));

        $rows = $qb -> get() -> getResultArray();

        /* ===========================
          SAVE VARIATIONS
          =========================== */

        $this -> db
                -> table("_product_variation")
                -> where("product_id", $productId)
                -> delete();

        $pos   = 1;
        $added = [];

        foreach ($rows as $r) {
            $this -> db -> table("_product_variation") -> insert([
                "product_id"           => $productId,
                "variation_product_id" => (int) $r["product_id"],
                "variation_name"       => $r["product_name"],
                "position"             => $pos++,
            ]);

            $added[] = (int) $r["product_id"];
        }

        return [
            "ok"                 => true,
            "product_id"         => $productId,
            "needle"             => $needle,
            "count"              => count($added),
            "variation_products" => $added,
        ];
    }

    private function smartSubstr(string $value, int $maxTokens = 5): string {
        $value = mb_strtolower($value);
        $value = preg_replace("/[^\p{L}\p{N}\s]+/u", " ", $value);

        $tokens = preg_split("/\s+/u", trim($value));
        $core   = [];

        foreach ($tokens as $t) {
            if (mb_strlen($t) <= 2) {
                continue;
            }
            if (ctype_digit($t)) {
                continue;
            }
            if (preg_match('/^[a-z]{0,3}\d{3,}$/i', $t)) {
                continue;
            }

            $core[] = $t;

            if (count($core) >= $maxTokens) {
                break;
            }
        }

        return implode(" ", $core);
    }

    public function generateBatch() {
        if (!$this -> request -> isAJAX()) {
            return $this -> response -> setJSON(["ok" => false]);
        }

        $cfg = service("settings") -> get("App.general")["variation_generator"] ??
                [];

        if (empty($cfg) || empty($cfg["base_field"])) {
            return $this -> response -> setJSON([
                        "ok"  => false,
                        "msg" => "Variation generator config not set",
            ]);
        }

        $offset = (int) $this -> request -> getPost("offset");
        $limit  = (int) $this -> request -> getPost("limit") ?: 50;

        $products = $this -> db
                -> table("_product")
                -> select("product_id")
                -> orderBy("product_id", "ASC")
                -> limit($limit, $offset)
                -> get()
                -> getResultArray();

        $processed = 0;

        foreach ($products as $p) {
            $this -> generateForProduct((int) $p["product_id"]);
            $processed++;
        }

        return $this -> response -> setJSON([
                    "ok"        => true,
                    "processed" => $processed,
                    "offset"    => $offset,
                    "limit"     => $limit,
        ]);
    }

}
