<?php

/*
 * CKFinder
 * ========
 * https://ckeditor.com/ckfinder/
 * Copyright (c) 2007-2021, CKSource - Frederico Knabben. All rights reserved.
 *
 * The software, this file and its contents are subject to the CKFinder
 * License. Please read the license.txt file before using, installing, copying,
 * modifying or distribute this file or part of its contents. The contents of
 * this file is part of the Source Code of CKFinder.
 */

namespace CKSource\CKFinder\Command;

use CKSource\CKFinder\Acl\Permission;
use CKSource\CKFinder\Cache\CacheManager;
use CKSource\CKFinder\Config;
use CKSource\CKFinder\Error;
use CKSource\CKFinder\Event\AfterCommandEvent;
use CKSource\CKFinder\Event\CKFinderEvent;
use CKSource\CKFinder\Event\FileUploadEvent;
use CKSource\CKFinder\Exception\InvalidExtensionException;
use CKSource\CKFinder\Exception\InvalidNameException;
use CKSource\CKFinder\Exception\InvalidUploadException;
use CKSource\CKFinder\Filesystem\File\UploadedFile;
use Symfony\Component\HttpFoundation\File\UploadedFile as UploadedFileBase;
use CKSource\CKFinder\Filesystem\Folder\WorkingFolder;
use CKSource\CKFinder\Filesystem\Path;
use CKSource\CKFinder\Image;
use CKSource\CKFinder\Thumbnail\ThumbnailRepository;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\HttpFoundation\Request;

class FileUpload extends CommandAbstract {

    protected $requestMethod = Request::METHOD_POST;
    protected $requires      = [Permission::FILE_CREATE];

    public function execute(Request $request, WorkingFolder $workingFolder, EventDispatcher $dispatcher, Config $config, CacheManager $cache, ThumbnailRepository $thumbsRepository) {
        // #111 IE9 download JSON issue workaround
        if ($request -> get('asPlainText')) {
            $uploadEvents = [
                CKFinderEvent::AFTER_COMMAND_FILE_UPLOAD,
                CKFinderEvent::AFTER_COMMAND_QUICK_UPLOAD,
            ];

            foreach ($uploadEvents as $eventName) {
                $dispatcher -> addListener($eventName, function (AfterCommandEvent $event) {
                    $response = $event -> getResponse();
                    $response -> headers -> set('Content-Type', 'text/plain');
                });
            }
        }

        $uploaded = 0;

        $warningErrorCode = null;
        $upload           = $request -> files -> get('upload');

        if (null === $upload) {
            throw new InvalidUploadException();
        }

        // Extra check to handle older Symfony components on PHP 8.1+.
        if (is_array($upload)) {
            $upload = new UploadedFileBase($upload['tmp_name'], $upload['name'], $upload['type'], $upload['error'], false);
        }

        $uploadedFile = new UploadedFile($upload, $this -> app);

        if (!$uploadedFile -> isValid()) {
            throw new InvalidUploadException($uploadedFile -> getErrorMessage());
        }

        $uploadedFile -> sanitizeFilename();

        if ($uploadedFile -> wasRenamed()) {
            $warningErrorCode = Error::UPLOADED_INVALID_NAME_RENAMED;
        }

        if (!$uploadedFile -> hasValidFilename() || $uploadedFile -> isHiddenFile()) {
            throw new InvalidNameException();
        }

        if (!$uploadedFile -> hasAllowedExtension()) {
            throw new InvalidExtensionException();
        }

        // Autorename if required
        $overwriteOnUpload = $config -> get('overwriteOnUpload');
        if (!$overwriteOnUpload && $uploadedFile -> autorename()) {
            $warningErrorCode = Error::UPLOADED_FILE_RENAMED;
        }

        $fileName = $uploadedFile -> getFilename();
        $fileName = $this -> transliterateFileName($fileName);

        if (!$uploadedFile -> isAllowedHtmlFile() && $uploadedFile -> containsHtml()) {
            throw new InvalidUploadException('HTML detected in disallowed file type', Error::UPLOADED_WRONG_HTML_FILE);
        }

        if ($config -> get('secureImageUploads') && $uploadedFile -> isImage() && !$uploadedFile -> isValidImage()) {
            throw new InvalidUploadException('Invalid upload: corrupted image', Error::UPLOADED_CORRUPT);
        }

        $maxFileSize = $workingFolder -> getResourceType() -> getMaxSize();

        if (!$config -> get('checkSizeAfterScaling') && $maxFileSize && $uploadedFile -> getSize() > $maxFileSize) {
            throw new InvalidUploadException('Uploaded file is too big', Error::UPLOADED_TOO_BIG);
        }

        if (Image::isSupportedExtension($uploadedFile -> getExtension())) {
            $imagesConfig = $config -> get('images');
            $image        = Image::create($uploadedFile -> getContents());

            if ($imagesConfig['maxWidth'] && $image -> getWidth() > $imagesConfig['maxWidth'] ||
                    $imagesConfig['maxHeight'] && $image -> getHeight() > $imagesConfig['maxHeight']) {
                $image -> resize($imagesConfig['maxWidth'], $imagesConfig['maxHeight'], $imagesConfig['quality']);
                $imageData = $image -> getData();
                $uploadedFile -> save($imageData);
            }

            $cache -> set(
                    Path::combine(
                            $workingFolder -> getResourceType() -> getName(),
                            $workingFolder -> getClientCurrentFolder(),
                            $fileName
                    ),
                            $image -> getInfo()
            );

            unset($imageData, $image);
        }

        if ($maxFileSize && $uploadedFile -> getSize() > $maxFileSize) {
            throw new InvalidUploadException('Uploaded file is too big', Error::UPLOADED_TOO_BIG);
        }

        $event = new FileUploadEvent($this -> app, $uploadedFile);
        $dispatcher -> dispatch($event, CKFinderEvent::FILE_UPLOAD);

        if (!$event -> isPropagationStopped()) {
//            $uploadedFileStream = $uploadedFile -> getContentsStream();
//            $uploaded           = (int) $workingFolder -> putStream($fileName, $uploadedFileStream, $uploadedFile -> getMimeType());
//
//            if (\is_resource($uploadedFileStream)) {
//                fclose($uploadedFileStream);
// 
//                       }
// >>> WEBP CONVERT BEGIN (JPG/PNG -> WEBP, overwrite original name & mime)
            $origExt       = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
            $shouldConvert = in_array($origExt, ['jpg', 'jpeg', 'png'], true);

            $targetMime   = $uploadedFile -> getMimeType();
            $targetName   = $fileName;
            $uploadStream = null;

// при нужда – качество от конфиг, иначе 85
            $webpQuality = 75;
            try {
                $q = $config -> get('ImageWebp.quality');
                if (is_numeric($q)) {
                    $webpQuality = max(0, min(100, (int) $q));
                }
            } catch (\Throwable $e) {
                // ignore
            }

            if ($shouldConvert) {
                // ново име: *.webp
                $base       = pathinfo($fileName, PATHINFO_FILENAME);
                $targetName = $base . '.webp';
                $targetMime = 'image/webp';

                // взимаме съдържанието на качения файл като бинар
                $binary = $uploadedFile -> getContents();

                // ще пишем в temp stream (php://temp), за да го подадем към putStream()
                $tmp       = fopen('php://temp', 'w+');
                $converted = false;

                // 1) Предпочитаме Imagick
                if (class_exists('\Imagick')) {
                    try {
                        $im = new \Imagick();
                        $im -> readImageBlob($binary);
                        $im -> setImageFormat('webp');
                        $im -> setImageCompressionQuality($webpQuality);
                        // пазим alpha при PNG
                        if (in_array($origExt, ['png'], true)) {
                            $im -> setImageAlphaChannel(\Imagick::ALPHACHANNEL_ACTIVATE);
                        }
                        $blob      = $im -> getImagesBlob();
                        fwrite($tmp, $blob);
                        $converted = true;
                        $im -> clear();
                        $im -> destroy();
                    } catch (\Throwable $e) {
                        $converted = false;
                    }
                }

                // 2) Fallback: GD
                if (!$converted && function_exists('imagewebp')) {
                    $img = @imagecreatefromstring($binary);
                    if ($img) {
                        // подобрения за PNG
                        if ($origExt === 'png') {
                            if (function_exists('imagepalettetotruecolor'))
                                @imagepalettetotruecolor($img);
                            if (function_exists('imagesavealpha'))
                                @imagesavealpha($img, true);
                        }
                        // imagewebp приема файл път, не stream → ще ползваме temp файл
                        $tmpFile = tempnam(sys_get_temp_dir(), 'ckf_webp_');
                        if (@imagewebp($img, $tmpFile, $webpQuality)) {
                            $webpBin = @file_get_contents($tmpFile);
                            if ($webpBin !== false) {
                                fwrite($tmp, $webpBin);
                                $converted = true;
                            }
                        }
                        @imagedestroy($img);
                        @unlink($tmpFile);
                    }
                }

                if ($converted) {
                    rewind($tmp);
                    $uploadStream = $tmp;
                } else {
                    // ако не успеем да конвертираме – качваме оригинала
                    if (is_resource($tmp))
                        fclose($tmp);
                    $uploadStream = $uploadedFile -> getContentsStream();
                    $targetName   = $fileName;
                    $targetMime   = $uploadedFile -> getMimeType();
                }
            } else {
                // други типове – качваме както са
                $uploadStream = $uploadedFile -> getContentsStream();
                $targetName   = $fileName;
                $targetMime   = $uploadedFile -> getMimeType();
            }

            // финално качване (вече с коректно име и mime)
            $uploaded = (int) $workingFolder -> putStream($targetName, $uploadStream, $targetMime);

            // чистене
            if (\is_resource($uploadStream)) {
                fclose($uploadStream);
            }

            // актуализираме работното име, ако е сменено
            $fileName = $targetName;
            // >>> WEBP CONVERT END


            if ($overwriteOnUpload) {
                $thumbsRepository -> deleteThumbnails(
                        $workingFolder -> getResourceType(),
                        $workingFolder -> getClientCurrentFolder(),
                        $fileName
                );
            }

            if (!$uploaded) {
                $warningErrorCode = Error::ACCESS_DENIED;
            }
        }

        $responseData = [
            'fileName' => $fileName,
            'uploaded' => $uploaded,
        ];

        if ($warningErrorCode) {
            $errorMessage          = $this -> app['translator'] -> translateErrorMessage($warningErrorCode, ['name' => $fileName]);
            $responseData['error'] = [
                'number'  => $warningErrorCode,
                'message' => $errorMessage,
            ];
        }

        return $responseData;
    }

    public static function transliterateFileName(string $name) {
        // Транслитерация: кирилица → латиница
        $cyr  = [
            'а', 'б', 'в', 'г', 'д', 'е', 'ж', 'з', 'и', 'й',
            'к', 'л', 'м', 'н', 'о', 'п', 'р', 'с', 'т', 'у',
            'ф', 'х', 'ц', 'ч', 'ш', 'щ', 'ъ', 'ь', 'ю', 'я',
            'А', 'Б', 'В', 'Г', 'Д', 'Е', 'Ж', 'З', 'И', 'Й',
            'К', 'Л', 'М', 'Н', 'О', 'П', 'Р', 'С', 'Т', 'У',
            'Ф', 'Х', 'Ц', 'Ч', 'Ш', 'Щ', 'Ъ', 'Ь', 'Ю', 'Я'
        ];
        $lat  = [
            'a', 'b', 'v', 'g', 'd', 'e', 'zh', 'z', 'i', 'y',
            'k', 'l', 'm', 'n', 'o', 'p', 'r', 's', 't', 'u',
            'f', 'h', 'ts', 'ch', 'sh', 'sht', 'a', '', 'yu', 'ya',
            'A', 'B', 'V', 'G', 'D', 'E', 'Zh', 'Z', 'I', 'Y',
            'K', 'L', 'M', 'N', 'O', 'P', 'R', 'S', 'T', 'U',
            'F', 'H', 'Ts', 'Ch', 'Sh', 'Sht', 'A', '', 'Yu', 'Ya'
        ];
        $name = str_replace($cyr, $lat, $name);

        // Заместване на интервали с "_"
        $name = preg_replace('/\s+/', '_', $name);

        // Премахване на всички символи, освен букви, цифри, _ и -
        $name = preg_replace('/[^a-zA-Zа-яА-Я0-9\.\-_]/u', '', $name);

        // Премахване на излишни последователни "_"
        $name = preg_replace('/_+/', '_', $name);

        return trim($name, '_');
    }

}
